# © 2024-2025 Danial Behzadi <dani.behzi@ubuntu.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# SPDX-License-Identifier: GPL-3.0-or-later

"""
Things to do with control socket
"""

from os.path import join

from gi.repository import GLib
from stem import Signal
from stem.connection import connect
from stem.control import Controller


def _get_controller() -> Controller:
    """
    Return the control socket
    """
    socket = join(GLib.get_user_config_dir(), "tractor", "control.sock")
    controller = connect(control_socket=socket)
    return controller


def send_signal(signal: str) -> None:
    """
    Send a signal to the tor process
    """
    controller = _get_controller()
    if controller:
        match signal:
            case "term":
                controller.signal(Signal.TERM)
                controller.close()
            case "newnym":
                controller.signal(Signal.NEWNYM)
                controller.close()
            case _:
                controller.close()
                raise ValueError(f"Wrong signal '{signal}'.")


def get_listener(listener_type: str) -> int:
    """
    Get configuration from control socket
    """
    controller = _get_controller()
    if controller:
        value = controller.get_listeners(listener_type)
        controller.close()
        return value[0]
    raise ValueError("No listener.")


def get_pid() -> int:
    """
    Get pid of the tor process
    """
    controller = _get_controller()
    if controller:
        pid = controller.get_pid()
        controller.close()
        return pid
    return 0


def get_bridge() -> str:
    """
    Get the current using bridges
    """
    controller = _get_controller()
    if controller:
        if controller.get_conf("UseBridges"):
            bridge = controller.get_conf("Bridge")
            controller.close()
            return bridge
    return ""
