// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package slhdsa

import (
	"slices"
	"testing"
)

func TestToInt(t *testing.T) {
	for _, tc := range []struct {
		x    []byte
		n    uint32
		want uint64
	}{
		{
			x:    []byte{0xF1},
			n:    1,
			want: 0xF1,
		},
		{
			x:    []byte{0xF1, 0xAB},
			n:    2,
			want: 0xF1AB,
		},
		{
			x:    []byte{0xF1, 0xAB, 0x03},
			n:    3,
			want: 0xF1AB03,
		},
		{
			x:    []byte{0xF1, 0xAB, 0x03, 0xCD},
			n:    4,
			want: 0xF1AB03CD,
		},
		{
			x:    []byte{0xF1, 0xAB, 0x03, 0xCD, 0xEF},
			n:    5,
			want: 0xF1AB03CDEF,
		},
		{
			x:    []byte{0xF1, 0xAB, 0x03, 0xCD, 0xEF, 0x01},
			n:    6,
			want: 0xF1AB03CDEF01,
		},
		{
			x:    []byte{0xF1, 0xAB, 0x03, 0xCD, 0xEF, 0x01, 0xEF},
			n:    7,
			want: 0xF1AB03CDEF01EF,
		},
		{
			x:    []byte{0xF1, 0xAB, 0x03, 0xCD, 0xEF, 0x01, 0xEF, 0x3F},
			n:    8,
			want: 0xF1AB03CDEF01EF3F,
		},
	} {
		if got := toInt(tc.x, tc.n); got != tc.want {
			t.Errorf("toInt(%v, %v) = %v, want %v", tc.x, tc.n, got, tc.want)
		}
	}
}

func TestToByte(t *testing.T) {
	for _, tc := range []struct {
		x    uint32
		n    uint32
		want []byte
	}{
		{
			x:    0xF1,
			n:    1,
			want: []byte{0xF1},
		},
		{
			x:    0xF1AB,
			n:    2,
			want: []byte{0xF1, 0xAB},
		},
		{
			x:    0xF1AB03,
			n:    3,
			want: []byte{0xF1, 0xAB, 0x03},
		},
		{
			x:    0xF1AB03CD,
			n:    4,
			want: []byte{0xF1, 0xAB, 0x03, 0xCD},
		},
	} {
		if got := toByte(tc.x, tc.n); !slices.Equal(got, tc.want) {
			t.Errorf("toByte(%v, %v) = %v, want %v", tc.x, tc.n, got, tc.want)
		}
	}
}

func TestBase2b(t *testing.T) {
	for _, tc := range []struct {
		x      []byte
		b      uint32
		outLen uint32
		want   []uint32
	}{
		// Test vectors generated using the reference implementation (https://github.com/sphincs/sphincsplus).
		{
			x:      []byte{0x10, 0xc0},
			b:      4,
			outLen: 3,
			want:   []uint32{0x1, 0x0, 0xc},
		},
		{
			x:      []byte{0x24, 0xc9, 0x33, 0x1f, 0xd2, 0x80, 0xe6, 0x45, 0xcf, 0xc, 0x2c, 0x4d, 0x6b, 0x42, 0x46, 0x53},
			b:      4,
			outLen: 32,
			want:   []uint32{0x2, 0x4, 0xc, 0x9, 0x3, 0x3, 0x1, 0xf, 0xd, 0x2, 0x8, 0x0, 0xe, 0x6, 0x4, 0x5, 0xc, 0xf, 0x0, 0xc, 0x2, 0xc, 0x4, 0xd, 0x6, 0xb, 0x4, 0x2, 0x4, 0x6, 0x5, 0x3},
		},
		{
			x:      []byte{0xeb, 0x4a, 0xa8, 0x31, 0xbe, 0x83, 0x5b, 0x70, 0xae, 0xca, 0x5e, 0x17, 0x5, 0xea, 0xfd, 0xdd, 0x8f, 0xc2, 0x6, 0xa3, 0xe1, 0x84, 0x32, 0x3},
			b:      4,
			outLen: 48,
			want:   []uint32{0xe, 0xb, 0x4, 0xa, 0xa, 0x8, 0x3, 0x1, 0xb, 0xe, 0x8, 0x3, 0x5, 0xb, 0x7, 0x0, 0xa, 0xe, 0xc, 0xa, 0x5, 0xe, 0x1, 0x7, 0x0, 0x5, 0xe, 0xa, 0xf, 0xd, 0xd, 0xd, 0x8, 0xf, 0xc, 0x2, 0x0, 0x6, 0xa, 0x3, 0xe, 0x1, 0x8, 0x4, 0x3, 0x2, 0x0, 0x3},
		},
		{
			x:      []byte{0x55, 0xdf, 0x97, 0xab, 0x41, 0x40, 0xff, 0x9, 0xc4, 0x50, 0x33, 0xed, 0xd5, 0xc6, 0xd, 0x7, 0x21, 0x68, 0x24, 0x40, 0x18, 0x2c, 0xd7, 0x84, 0xa7, 0xfb, 0xb7, 0x36, 0xfd, 0x53, 0x54, 0xb4},
			b:      4,
			outLen: 64,
			want:   []uint32{0x5, 0x5, 0xd, 0xf, 0x9, 0x7, 0xa, 0xb, 0x4, 0x1, 0x4, 0x0, 0xf, 0xf, 0x0, 0x9, 0xc, 0x4, 0x5, 0x0, 0x3, 0x3, 0xe, 0xd, 0xd, 0x5, 0xc, 0x6, 0x0, 0xd, 0x0, 0x7, 0x2, 0x1, 0x6, 0x8, 0x2, 0x4, 0x4, 0x0, 0x1, 0x8, 0x2, 0xc, 0xd, 0x7, 0x8, 0x4, 0xa, 0x7, 0xf, 0xb, 0xb, 0x7, 0x3, 0x6, 0xf, 0xd, 0x5, 0x3, 0x5, 0x4, 0xb, 0x4},
		},
		{
			x:      []byte{0xb2, 0xc8, 0x73, 0x76, 0x9f, 0x90, 0x8c, 0x51, 0x20, 0x59, 0x7c, 0xe, 0x3f, 0x21, 0xc0, 0xf1, 0xa8, 0xb0, 0xc8, 0x59, 0xc, 0x3d, 0xaf, 0x23, 0x1b},
			b:      6,
			outLen: 33,
			want:   []uint32{0x2c, 0x2c, 0x21, 0x33, 0x1d, 0x29, 0x3e, 0x10, 0x23, 0x5, 0x4, 0x20, 0x16, 0x17, 0x30, 0xe, 0xf, 0x32, 0x7, 0x0, 0x3c, 0x1a, 0x22, 0x30, 0x32, 0x5, 0x24, 0xc, 0xf, 0x1a, 0x3c, 0x23, 0x6},
		},
		{
			x:      []byte{0x95, 0x95, 0xea, 0x4d, 0xb1, 0x29, 0x8e, 0x63, 0x9a, 0xcb, 0x8b, 0x90, 0xdf, 0x47, 0x85, 0x10, 0x79, 0x92, 0xd0, 0x2f, 0x8, 0x78, 0x3e, 0xc3, 0xe0, 0xbd, 0x38, 0xda, 0x61, 0x93, 0x81, 0x2c, 0xd4},
			b:      8,
			outLen: 33,
			want:   []uint32{0x95, 0x95, 0xea, 0x4d, 0xb1, 0x29, 0x8e, 0x63, 0x9a, 0xcb, 0x8b, 0x90, 0xdf, 0x47, 0x85, 0x10, 0x79, 0x92, 0xd0, 0x2f, 0x8, 0x78, 0x3e, 0xc3, 0xe0, 0xbd, 0x38, 0xda, 0x61, 0x93, 0x81, 0x2c, 0xd4},
		},
		{
			x:      []byte{0x58, 0x51, 0x57, 0x6d, 0x43, 0x1, 0x6c, 0x66, 0xae, 0xe4, 0x83, 0xa6, 0x97, 0xe6, 0xe0, 0xf4, 0x4d, 0xe6, 0x93, 0xc6, 0x2c, 0xf6, 0x46, 0x82, 0x6e, 0xdb, 0x0, 0x72, 0x35, 0xb5, 0x23, 0x9, 0xfa, 0x2a, 0x91, 0x9e, 0x3b, 0xc4, 0x86, 0x80},
			b:      9,
			outLen: 35,
			want:   []uint32{0xb0, 0x145, 0xbb, 0xd4, 0x60, 0x5b, 0x33, 0xae, 0x1c9, 0xe, 0x134, 0x17e, 0xdc, 0x3d, 0x26, 0x1e6, 0x127, 0x118, 0x167, 0x164, 0xd0, 0x9b, 0x16d, 0x100, 0xe4, 0xd6, 0x1a9, 0x30, 0x13f, 0x8a, 0x148, 0x19e, 0x77, 0x112, 0x34},
		},
		{
			x:      []byte{0x60, 0xf2, 0xcf, 0xe4, 0xa4, 0xc, 0x28, 0xa1, 0x78, 0xad, 0x33, 0xd3, 0x71, 0x62, 0x13, 0x3, 0x8e, 0xf6, 0x1d, 0x69, 0xf5},
			b:      12,
			outLen: 14,
			want:   []uint32{0x60f, 0x2cf, 0xe4a, 0x40c, 0x28a, 0x178, 0xad3, 0x3d3, 0x716, 0x213, 0x38, 0xef6, 0x1d6, 0x9f5},
		},
		{
			x:      []byte{0xc5, 0xc6, 0x68, 0x70, 0x9, 0xe0, 0xd9, 0xd3, 0x6c, 0x8d, 0x2f, 0x77, 0x5b, 0x45, 0xde, 0x58, 0xca, 0x64, 0x34, 0xb1, 0xc7, 0xfe, 0x27, 0xdf, 0x6b, 0xc4, 0x1c, 0x13, 0xd3, 0x34},
			b:      14,
			outLen: 17,
			want:   []uint32{0x3171, 0x2687, 0x27, 0x20d9, 0x34db, 0x8d2, 0x3ddd, 0x1b45, 0x3796, 0xca6, 0x10d2, 0x31c7, 0x3f89, 0x3df6, 0x2f10, 0x1c13, 0x34cd},
		},
		{
			x:      []byte{0x7c, 0x4, 0xea, 0x2, 0xf6, 0x30, 0xc8, 0x68, 0xa2, 0x95, 0x37, 0xfe, 0x89, 0x2c, 0x38, 0x35, 0xf1, 0x8e, 0x4f, 0xb5, 0x78, 0x7b, 0xdc, 0xfd, 0xb, 0x24, 0xc1, 0x9b, 0xf6, 0x4b, 0xec, 0x6f, 0x4f, 0xc4, 0xe9, 0x56, 0xb, 0x79, 0x70},
			b:      14,
			outLen: 22,
			want:   []uint32{0x1f01, 0xea0, 0xbd8, 0x30c8, 0x1a28, 0x2953, 0x1ffa, 0x92c, 0xe0d, 0x1f18, 0x393e, 0x3578, 0x1ef7, 0xfd0, 0x2c93, 0x19b, 0x3d92, 0x3ec6, 0x3d3f, 0x4e9, 0x1582, 0x3797},
		},
	} {
		if got := base2b(tc.x, tc.b, tc.outLen); !slices.Equal(got, tc.want) {
			t.Errorf("base2b(%v, %v, %v) = %v, want %v", tc.x, tc.b, tc.outLen, got, tc.want)
		}
	}
}
